use crate::{Hash, hash};

use serde::{Deserialize, Serialize};
use serde_big_array::BigArray;

#[derive(Serialize)]
pub enum PathEnc<'a, T: Serialize> {
    Leaf(&'a T),
    Node(Hash, Hash),
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum Dir {
    Lhs(Hash),
    Rhs(Hash),
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Path<T: Serialize, const L: usize> {
    leaf: T,
    #[serde(with = "BigArray")]
    path: [Dir; L],
}

impl<T: Serialize, const L: usize> Path<T, L> {
    pub fn verify(self, root: Hash, idx: usize) -> Result<T, anyhow::Error> {
        anyhow::ensure!(idx < (1 << L), "index out of bounds");
        let mut pos = 0;
        let mut hsh = hash(&PathEnc::Leaf::<T>(&self.leaf));
        for (lvl, dir) in self.path.into_iter().enumerate() {
            match dir {
                Dir::Lhs(rhs) => {
                    hsh = hash(&PathEnc::Node::<T>(hsh, rhs));
                }
                Dir::Rhs(lhs) => {
                    hsh = hash(&PathEnc::Node::<T>(lhs, hsh));
                    pos |= 1 << lvl;
                }
            }
        }
        anyhow::ensure!(pos == idx, "index does not match");
        anyhow::ensure!(hsh == root, "root hash does not match");
        Ok(self.leaf)
    }
}

#[derive(Debug)]
pub enum Tree<T: Clone + Serialize> {
    Leaf {
        root: Hash,
        leaf: T,
    },
    Node {
        root: Hash,
        size: usize,
        lhs: Box<Tree<T>>,
        rhs: Box<Tree<T>>,
    },
}

impl<T: Clone + Serialize> Tree<T> {
    pub fn size(&self) -> usize {
        match self {
            Tree::Leaf { .. } => 1,
            Tree::Node { size, .. } => *size,
        }
    }

    pub fn root(&self) -> Hash {
        match self {
            Tree::Leaf { root, .. } => *root,
            Tree::Node { root, .. } => *root,
        }
    }

    pub fn new(elems: &[T]) -> Self {
        assert!(!elems.is_empty());
        assert!(elems.len().is_power_of_two());
        if elems.len() == 1 {
            let leaf = elems[0].clone();
            Tree::Leaf {
                root: hash(&PathEnc::Leaf::<T>(&leaf)),
                leaf,
            }
        } else {
            assert_eq!(elems.len() % 2, 0);
            let (lhs, rhs) = elems.split_at(elems.len() / 2);
            let lhs = Box::new(Self::new(lhs));
            let rhs = Box::new(Self::new(rhs));
            Tree::Node {
                root: hash(&PathEnc::Node::<T>(lhs.root(), rhs.root())),
                size: lhs.size() + rhs.size(),
                lhs,
                rhs,
            }
        }
    }

    pub fn open<const L: usize>(&self, pos: usize) -> Path<T, L> {
        fn walk<T: Clone + Serialize>(tree: &Tree<T>, pos: usize) -> (T, Vec<Dir>) {
            assert!(pos < tree.size());
            match tree {
                Tree::Leaf { leaf, .. } => (leaf.clone(), vec![]),
                Tree::Node { lhs, rhs, .. } => {
                    if pos < lhs.size() {
                        let mut res = walk(lhs, pos);
                        res.1.push(Dir::Lhs(rhs.root()));
                        res
                    } else {
                        let mut res = walk(rhs, pos - lhs.size());
                        res.1.push(Dir::Rhs(lhs.root()));
                        res
                    }
                }
            }
        }
        assert!(pos < self.size());
        assert_eq!(1 << L, self.size());
        let (leaf, path) = walk(self, pos);
        Path {
            leaf,
            path: path.try_into().unwrap(),
        }
    }
}

#[test]
fn test_merkle_tree() {
    fn test_depth<const L: usize>() {
        let elems: Vec<i32> = (1..=(1 << L)).collect();
        let tree = Tree::<i32>::new(&elems);
        let root = tree.root();
        for (pos, elm) in elems.into_iter().enumerate() {
            let path: Path<i32, L> = tree.open(pos);
            assert_eq!(path.verify(root, pos).unwrap(), elm);
        }
    }
    test_depth::<0>();
    test_depth::<1>();
    test_depth::<2>();
    test_depth::<3>();
    test_depth::<10>();
}
