use rug::Integer;

use crate::{
    CHAL_BYTES, CS, Expr, Prefix, Proof, QUERIES, SEP_CHALLG, SEP_PREFIX, SEP_PROOFS, SEP_ROUND1,
    SEP_ROUND2, affine, challenge_idx, challenge_int, check_combination_norm, primes::PRIMES,
};

#[derive(Default)]
pub struct Verifier {
    vars: Vec<Expr>,
    zero: Vec<Expr>,
}

impl CS for Verifier {
    fn var(&mut self, _value: Option<Integer>) -> Result<Expr, anyhow::Error> {
        let var = Expr::var(self.vars.len());
        self.vars.push(var.clone());
        Ok(var)
    }

    fn zero(&mut self, expr: Expr) -> Result<(), anyhow::Error> {
        self.zero.push(expr.clone());
        Ok(())
    }
}

impl Verifier {
    pub fn verify(mut self, pf: Proof, msg: &[u8]) -> Result<(), anyhow::Error> {
        // create a domain seperated transcript
        let mut tx = merlin::Transcript::new(SEP_PROOFS);
        tx.append_message(
            SEP_PREFIX,
            &bincode::serialize(&Prefix {
                msg: msg.to_vec(),
                vars: self.vars.len(),
                zero: self.zero.iter().map(|expr| expr.hash()).collect(),
            })?,
        );

        // round 1: receieve the commitment to the variable assignments
        // squeeze the linear combination
        tx.append_message(SEP_ROUND1, &bincode::serialize(&pf.root)?);
        let challenges: Vec<Integer> = (0..self.vars.len())
            .map(|_| challenge_int(&mut tx, SEP_CHALLG, CHAL_BYTES))
            .collect();

        // round 2: receive the random linear combination of variables
        tx.append_message(SEP_ROUND2, &bincode::serialize(&pf.rand)?);
        check_combination_norm(&pf.rand, self.vars.len())?;
        self.zero(affine(
            challenges.into_iter().zip(self.vars.clone().into_iter()),
            -pf.rand,
        ))?;

        // sample opening positions
        let poss = [(); QUERIES].map(|_| challenge_idx(&mut tx, SEP_CHALLG, 0..PRIMES.len()));

        // verify all Merkle proofs
        let mut leafs = Vec::new();
        for (i, pi) in poss.iter().copied().zip(pf.open.0) {
            let leaf = pi.verify(pf.root, i)?;
            anyhow::ensure!(leaf.len() == self.vars.len());
            leafs.push(leaf);
        }

        // check at every sampled position
        for (leaf_i, prime_i) in poss.iter().enumerate() {
            for expr in self.zero.iter() {
                anyhow::ensure!(
                    expr.eval(&leafs[leaf_i], PRIMES[*prime_i])? == 0,
                    "zero expression failed at position {}",
                    leaf_i
                );
            }
        }

        // looks good :)
        Ok(())
    }
}
