use axum::{
    extract::Json,
    http::StatusCode,
    response::Html,
    routing::{get, post},
    Router,
};
use ecpf::{Expr, Proof, Verifier, CS};
use rug::Integer;
use serde::{Deserialize, Serialize};
use tokio;

const PAGE: &str = include_str!("page.html");

// relation to prove: some elliptic curve
fn relation<C: CS>(cs: &mut C, wit: Option<(Integer, Integer)>) -> Result<(), anyhow::Error> {
    let x = cs.var(wit.as_ref().map(|w| w.0.clone()))?;
    let y = cs.var(wit.as_ref().map(|w| w.1.clone()))?;
    let curve_a: i32 = -567;
    let curve_b: i32 = -5346;
    let y2 = y.clone() * y.clone();
    let x2 = x.clone() * x.clone();
    let x3 = x2.clone() * x.clone();
    cs.eq(y2, x3 + Expr::cnst(curve_a) * x + Expr::cnst(curve_b))
}

#[derive(Serialize, Deserialize, Clone)]
struct Solution {
    proof: Proof,
    name: String,
}

fn verify(req: Solution) -> Result<String, anyhow::Error> {
    let mut verifier = Verifier::default();
    relation(&mut verifier, None)?;
    verifier.verify(req.proof, req.name.as_ref())?;
    Ok(req.name)
}

async fn serve_page() -> Html<&'static str> {
    Html(PAGE)
}

#[derive(Serialize)]
enum Solve {
    Failed,
    Success { name: String, message: String },
}

async fn handle_solve(
    axum::extract::Extension(flag): axum::extract::Extension<String>,
    Json(req): Json<Solution>
) -> Result<Json<Solve>, StatusCode> {
    match verify(req) {
        Ok(name) => Ok(Json(Solve::Success {
            name,
            message: flag,
        })),
        Err(_) => Ok(Json(Solve::Failed)),
    }
}

#[tokio::main]
async fn main() {
    // Check that FLAG environment variable exists at startup
    let flag = std::env::var("FLAG").unwrap_or_else(|_| {
        eprintln!("WARNING: FLAG environment variable not set, using default");
        "FLAG environment variable not set".to_string()
    });
    println!("Server starting with flag of {} characters", flag.len());

    // create axum router
    let app = Router::new()
        .route("/solve", post(handle_solve))
        .route("/", get(serve_page))
        .layer(axum::extract::Extension(flag));

    // define server address and create listener
    println!("Server listening on 0.0.0.0:3000");
    let listener = tokio::net::TcpListener::bind("0.0.0.0:3000").await.unwrap();

    // start server
    axum::serve(listener, app).await.unwrap();
}
